package com.objectriver.microservices.things.json;

import com.objectriver.microservices.things.RestDocument;
import com.objectriver.microservices.things.RestList;
import com.objectriver.microservices.things.RestObject;
import com.objectriver.microservices.things.RestThingType;
import com.objectriver.microservices.things.abstracts.RestScope;
import com.objectriver.microservices.things.abstracts.RestThing;
import org.apache.http.HttpResponse;

import javax.json.Json;
import javax.json.stream.JsonLocation;
import javax.json.stream.JsonParser;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.Stack;

/**
 *                             NOTICE
 *               COPYRIGHT (c) 2016 ObjectRiver Inc
 *                UNPUBLISHED - ALL RIGHTS RESERVED
 */
public class RestJsonParser implements JsonParser {
    protected final JsonParser parser;
    public RestJsonParser(String string ) {
        InputStream input = new ByteArrayInputStream(string.getBytes(StandardCharsets.UTF_8));
        parser = Json.createParser(input);
    }
    public RestJsonParser(HttpResponse response ) throws IOException {
        parser = Json.createParser(response.getEntity().getContent());
    }
    public RestJsonParser(InputStream input ) {
        parser = Json.createParser(input);
    }
    static public class RestJsonParserException extends Exception {
        public RestJsonParserException(String msg) { super(msg); }
        public RestJsonParserException(Throwable cause) { super(cause); }
        public RestJsonParserException(String msg, Throwable cause) {
            super(msg,cause);
        }
    }

    /**
     * The following just handle differnt input formats.
     */
    public static <T extends RestScope> T parseJsonObject(T scope, HttpResponse response) throws RestJsonParser.RestJsonParserException {
        try {
            return new RestJsonParser(response.getEntity().getContent()).parseObject(scope);
        }
        catch(IOException ex) {
            throw new RestJsonParser.RestJsonParserException(ex);
        }
    }
    public static <T extends RestScope> T parseJsonObject(T scope, InputStream input) throws RestJsonParser.RestJsonParserException {
        try {
            return new RestJsonParser(input).parseObject(scope);
        }
        finally{}
    }
    public static <T extends RestScope> T parseJsonObject(T scope, String input) throws RestJsonParser.RestJsonParserException {
        try {
            return new RestJsonParser(input).parseObject(scope);
        }
        finally{}
    }
    /**
     * This method can parse either a CaiJsonDocument or a CaiJsonObject
     */
    public <T extends RestScope> T parseObject(T obj) throws RestJsonParserException {
        if(!this.hasNext()) throw new RestJsonParserException("Nothing to parser!");
        Event event = this.peek();
        if(event==Event.START_OBJECT) {
            this.next(); //consume start_object
            while(this.hasNext() && (event=this.peek())!=Event.END_OBJECT) {
                if(event==Event.KEY_NAME) {
                    this.parseElement(obj);
                }
                else throw new RestJsonParserException("\"Object Parser expected key! Found " + event.toString());
            }
            if(this.hasNext()) this.next(); //consume peek END_OBJECT;
        }
        else throw new RestJsonParserException("Parser could not find START_OBJECT");
        return obj;
    }
//    public <T extends RestThing> T parseDictionary(T obj) throws RestJsonParserException {
//        if(!this.hasNext()) throw new RestJsonParserException("Nothing to parser!");
//        Event event = this.next();
//        if(event==Event.START_OBJECT) {
//            while(this.hasNext() && (event=this.peek())!=Event.END_OBJECT) {
//                if(event==Event.KEY_NAME) {
//                    this.parseElement(obj);
//                }
//                else throw new RestJsonParserException("\"Object Parser expected key! Found " + event.toString());
//            }
//            if(this.hasNext()) this.next(); //consume peek END_OBJECT;
//        }
//        else throw new RestJsonParserException("Parser could not find START_OBJECT");
//        return obj;
//    }
    public <T extends RestThing> RestList<T> parseList(RestList<T> list, T concrete) throws RestJsonParserException {
        if(!this.hasNext()) throw new RestJsonParserException("Nothing to parser!");
        Event event = this.next();
        if(event==Event.START_ARRAY) {
            while(this.hasNext() && (event=this.peek())!=Event.END_ARRAY) {
                switch(event) {
                    case START_OBJECT: {
                        try {
                            T obj = (T)concrete.getClass().getConstructor(String.class).newInstance(concrete.identifier);
                            obj.setParent(list);
                            list.add(obj);
                            list.setOfType(RestThingType.Object);
                            parseObject((RestScope)obj);
                        }
                        catch(NoSuchMethodException ignore){}
                        catch(IllegalAccessException ignore){}
                        catch(InvocationTargetException ignore){}
                        catch(InstantiationException ignore){}
                        break;
                    }
                    default:
                        throw new RestJsonParserException("Unexpected START_OBJECT Event!" + event.toString());
                }
            }
            if(this.hasNext()) this.next(); //consume peek END_ARRAY;
        }
        else throw new RestJsonParserException("Parser could not find START_OBJECT");
        return list;
    }
    public RestList parseList(RestList list) throws RestJsonParserException {
        if(!this.hasNext()) throw new RestJsonParserException("Nothing to parser!");
        Event event = this.next();
        if(event==Event.START_ARRAY) {
            while(this.hasNext() && (event=this.peek())!=Event.END_ARRAY) {
                switch(event) {
                    case VALUE_FALSE:
                        list.add(false);
                        list.setOfType(RestThingType.Boolean);
                        this.next(); //consume peek;
                        break;
                    case VALUE_NULL:
                        list.add(null);
                        this.next(); //consume peek;
                        break;
                    case VALUE_TRUE:
                        list.add(true);
                        list.setOfType(RestThingType.Boolean);
                        this.next(); //consume peek;
                        break;
                    case VALUE_STRING:
                        list.add(this.getString());
                        list.setOfType(RestThingType.String);
                        this.next(); //consume peek;
                        break;
                    case VALUE_NUMBER: {
                        BigDecimal big = this.getBigDecimal();
                        if(big.scale()==0) {
                            list.add(big.longValue());
                            list.setOfType(RestThingType.Integer);
                        }
                        else {
                            list.add(big.doubleValue());
                            list.setOfType(RestThingType.Decimal);
                        }
                        this.next(); //consume peek;
                        break;
                    }
                    case START_ARRAY: { // list of list
                        //todo:
//                        RestList obj = new RestList(key);
//                        obj.setParent(obj);
//                        parseList(obj);
                        break;
                    }
                    case START_OBJECT: {
                        RestObject obj = new RestObject("list_item");
                        obj.setParent(list);
                        list.add(obj);
                        list.setOfType(RestThingType.Object);

                        parseObject(obj);
                        break;
                    }
                    default:
                        throw new RestJsonParserException("Unexpected Event!" + event.toString());
                }
            }
            if(this.hasNext()) this.next(); //consume peek END_ARRAY;
        }
        else throw new RestJsonParserException("Parser could not find START_OBJECT");
        return list;
    }
    protected void parseElement(RestScope scope) throws RestJsonParserException {
        Event event = this.next(); // KEY_NAME;
        String key = this.getString();
        event = this.peek();
        switch(event) {
            case VALUE_FALSE:
                this.next(); //consume peek;
                scope.put(key,false);
                break;
            case VALUE_NULL:
                this.next(); //consume peek;
                scope.put(key,null);
                break;
            case VALUE_TRUE:
                this.next(); //consume peek;
                scope.put(key,true);
                break;
            case VALUE_STRING:
                this.next(); //consume peek;
                scope.put(key,this.getString());
                break;
            case VALUE_NUMBER:
                this.next(); //consume peek;
                scope.put(key,this.getBigDecimal());
                break;
            case START_ARRAY: {
                RestList list = new RestList(key);
                list.setParent(scope);
                parseList(list);
                scope.put(key,list);
                break;
            }
            case START_OBJECT: {
                RestObject obj = new RestObject(key);
                obj.setParent(scope);
                parseObject(obj);
                scope.put(key,obj);
                break;
            }
            default:
                throw new RestJsonParserException("Unexpected Event!" + event.toString());
        }
    }

    /**
     * JsonParser Implementation plus peek,lookahead logic.
     */
    protected Stack<Event> peekEventStack = new Stack<Event>();
    @Override public boolean hasNext() {
        if(!peekEventStack.empty())
            return true;
        return parser.hasNext();
    }
    public Event peek() {
        if(!peekEventStack.empty())
            return peekEventStack.peek();
        return peekEventStack.push(parser.next());
    }
    @Override public Event next() {
        if(!peekEventStack.empty()) {
            return peekEventStack.pop();
        }
        return parser.next();
    }
    public boolean lookahead(Event[] events) {
        for(int ii=1; ii<=events.length; ii++) {
            if(peekEventStack.size()<events.length) peekEventStack.push(parser.next()); // push enough events to satisfy lookahead.
            if(peekEventStack.search(events[ii-1])!=ii) return false;
        }
        return true;
    }
    @Override public String getString() {
        return parser.getString();
    }
    @Override public boolean isIntegralNumber() {
        return parser.isIntegralNumber();
    }
    @Override public int getInt() {
        return parser.getInt();
    }
    @Override public long getLong() {
        return parser.getLong();
    }
    @Override public BigDecimal getBigDecimal() {
        return parser.getBigDecimal();
    }
    @Override public JsonLocation getLocation() {
        return parser.getLocation();
    }
    @Override public void close() {
        parser.close();
    }

    public static void main(String[] args) {
        String json = "{\n" +
                "\t\"obj1\": {\n" +
                "\t\t\"int\": 123,\n" +
                "\t\t\"float\": 123.456,\n" +
                "\t\t\"char\": \"c\",\n" +
                "\t\t\"bool\": true,\n" +
                "\t\t\"string\": \"hello\",\n" +
                "\t\t\"obj2\": {\n" +
                "\t\t\t\"int2\": 2,\n" +
                "\t\t\t\"list\": [\n" +
                "\t\t\t\t123,\n" +
                "\t\t\t\t456,\n" +
                "\t\t\t\t789\n" +
                "\t\t\t],\n" +
                "\t\t\t\"string\": \"followlist\"\n" +
                "\t\t},\n" +
                "\t\t\"byte\": 12,\n" +
                "\t\t\"intnull\": null,\n" +
                "\t\t\"intnull2\": null,\n" +
                "\t\t\"nullobject\": {},\n" +
                "\t\t\"nulllist\": [],\n" +
                "\t\t\"emptyobject\": {}\n" +
                "\t}\n" +
                "}";
        try {
            InputStream stream = new ByteArrayInputStream(json.getBytes("UTF-8"));
            RestJsonParser jsonParser = new RestJsonParser(stream);
            RestDocument doc = new RestDocument();
            jsonParser.parseObject(doc);
            System.out.println("OK");
        }
        catch(UnsupportedEncodingException ex) {
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch(RestJsonParserException ex) {
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
    }
}
