package com.objectriver.microservices.rest;

import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpRequestBase;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.StringTokenizer;
import java.util.List;

/**
 * This class is the base abstract class for the applications client
 * but does contain the HTTP verb and metadata related to request and response.
 * This is just connection information.
 */
public abstract class RestHttpClient {
    protected HttpClient client = null;
    protected HttpHost host = null;
    protected ArrayList<Header> headers = new ArrayList();
    /**
     * Uri
     */
    protected URI uri;
    /**
     * Pieces
     */
    protected String webapp = null;
    protected String endpoint = "";

    public RestHttpClient(URI uri) throws RestException {
        setUri(uri);
    }

    /**
     * setUri() is used by protocols that return a new URL as part of Oauth2
     * protocol. i.e. SalesForce.
     */
    public URI getUri() {
        return uri;
    }
    public void setUri(URI uri) throws RestException {
        if(uri.getHost()==null) {
            throw new RestException("URI must be absolute!");
        }
        this.uri = uri;
        this.host = new HttpHost(uri.getHost(),uri.getPort(),uri.getScheme());

        StringTokenizer st = new StringTokenizer(uri.getPath(),"/");
        if(st.countTokens()==1) {
            this.webapp = st.nextToken() + "/";
        }
        else if(st.countTokens()>=2) {
            this.webapp = st.nextToken() + "/";
            this.endpoint = st.nextToken() + "/";
        }
        while(st.hasMoreTokens()) {
            this.endpoint += st.nextToken();
            if(st.hasMoreTokens()) this.endpoint += "/";
        }
//        String userInfo = uri.getUserInfo();
    }

    public RestHttpClient(RestHttpClient that) throws RestException {
        this.client = that.client;
        this.host = that.host;
        this.headers = that.headers;
        this.uri = that.uri;
        this.webapp = that.webapp;
        this.endpoint=that.endpoint;
    }

    public void addHeader(Header header) {
        if(header!=null) {
            headers.add(header);
        }
    }
    public void addHeaders(HttpRequestBase requestBase) {
        for(Header header : headers) {
            requestBase.addHeader(header);
        }
    }
    public List<Header> getAddedHeaders() {
        return headers;
    }

    public void setWebapp(String webapp) {
        if(webapp.endsWith("/")) this.webapp = webapp;
        else this.webapp = webapp + "/";
    }
    public String getWebapp() {
        return webapp;
    }

    public String getEndpoint() {
        return endpoint;
    }
    public void setEndpoint(String endpoint) {
        if(endpoint.endsWith("/")) this.endpoint = endpoint;
        else this.endpoint = endpoint + "/";
    }

    public void setClient(HttpClient client) {
        this.client=client;
    }
    public HttpClient getClient() {
        return client;
    }

    public HttpHost getHost() {
        return host;
    }

    /**
     * Uri properties
     */
    public String getScheme() { return uri.getScheme(); }
    public int getPort() { return uri.getPort(); }
    public String getPath() {
        return uri.getPath();
    }
    public String getQuery() {
        return uri.getQuery();
    }
    public String getSchemeSpecific() {
        return uri.getSchemeSpecificPart();
    }
    public String getFragment() {
        return uri.getFragment();
    }


    public String toString() {
        String userinfo = uri.getUserInfo();
        String query = uri.getQuery();
        String fragment = uri.getFragment();
        String uristr =
                host.getSchemeName() + "//" +
                ((userinfo==null)?"":userinfo + "%40") +
                host.getHostName() +
                ((host.getPort()==(-1))?"":":" + host.getPort() ) + "/" +
                ((this.webapp==null)?"":this.webapp) +
                ((this.endpoint==null)?"":this.endpoint) +
                ((query==null)?"":"?" + query) +
                ((fragment==null)?"":"#" + fragment)
                ;
        return uristr;
    }
    /**
     * Platform Sfdc workaround hack. method can not contain '."
     */
    private boolean salesforce = false;
    public boolean isSalesforce() { return salesforce; }
    public void setSalesforce(boolean isSfdc) {
        this.salesforce=isSfdc;
    }

}
