package com.objectriver.microservices.things.json;

import com.objectriver.microservices.rest.RestException;
import com.objectriver.microservices.things.*;
import com.objectriver.microservices.things.abstracts.RestScope;
import com.objectriver.microservices.things.abstracts.RestThing;
import com.objectriver.microservices.things.json.abstracts.RestThingVisitor;
import com.objectriver.microservices.util.ISO8601DateTimeSingleton;
import sun.reflect.generics.reflectiveObjects.NotImplementedException;

import java.io.IOException;
import java.io.OutputStream;
import java.sql.Date;
import java.sql.Timestamp;
import java.util.Iterator;
import java.util.List;

/**
 *                             NOTICE
 *               COPYRIGHT (c) 2016 ObjectRiver Inc
 *                UNPUBLISHED - ALL RIGHTS RESERVED
 */

public class RestJsonVisitor extends RestThingVisitor {
    protected StringBuilder buffer = null;
    protected RestException excep = null;
    protected OutputStream outStream = null;
    protected RestJsonVisitor() {
        buffer = new StringBuilder();
    }
    protected int tabs=0;

    /**
     * Encode Thing to String
     */
    public static String execute(RestThing thing)  throws RestException {
        RestJsonVisitor visitor = new RestJsonVisitor();
        thing.visit(visitor);
        if(visitor.excep!=null)
            throw visitor.excep;
        return visitor.buffer.toString();
    }
    /**
     * Encode Thing to Stream
     */
    public static void execute(RestThing thing, OutputStream stream) throws RestException {
        try {
            RestJsonVisitor visitor = new RestJsonVisitor();
            visitor.outStream = stream;
            thing.visit(visitor);
            if (visitor.excep != null)
                throw visitor.excep;
        }
        finally {
            try { stream.close();} catch (IOException ex) {}
        }
    }
    /** Document */
    public boolean beginVisitOfRestDocument(RestDocument value) {
        buffer.append('{').append('\n');
        tabs++;
        return true;
    }
    public void endVisitOfRestDocument(RestDocument value) {
        try {
            tabs--;
            int index = buffer.lastIndexOf(",");
            if(index!=(-1)) {
                buffer.deleteCharAt(index); // trailing ','
            }
            buffer.append('}').append('\n');
            if (outStream != null) {
                outStream.write(buffer.toString().getBytes("UTF-8"));
                buffer = new StringBuilder();
            }
        }
        catch(IOException ex) {
            excep = new RestException("IO Exception", ex);
        }
    }
    /** Object */
    public boolean beginVisitOfRestObject(RestObject value) {
        if(value instanceof RestDictionary) return true;
        tabs++;
        buffer.append('{').append('\n');
        return true;
    }
    public void endVisitOfRestObject(RestObject value) {
        if(value instanceof RestDictionary) return;
        try {
            if(!buffer.toString().endsWith("{\n")) {
                int index = buffer.lastIndexOf(",");
                if (index != (-1)) {
                    buffer.deleteCharAt(index); // trailing ','
                }
            }

            tabs--;
            insertTabs();
            buffer.append('}');
            if (outStream != null) {
                outStream.write(buffer.toString().getBytes("UTF-8"));
                buffer = new StringBuilder();
            }
        }
        catch(IOException ex) {
            excep = new RestException("IO Exception", ex);
        }
    }
    /** Scope */
    public boolean beginVisitOfRestScope(RestScope value) {
        return true;
    }
    public void endVisitOfRestScope(RestScope value) {
    }
    /** List */
    public boolean beginVisitOfRestList(RestList value) {
        tabs++;
        buffer.append('[').append('\n');
        Iterator iter = value.iterator();
        while(iter.hasNext()) {
            Object item = iter.next();
            insertTabs();
            if(item==null) {
                if(value.getType()== RestThingType.Object) {
                    buffer.append("{}");
                }
                if(value.getType()== RestThingType.List) {
                    buffer.append("[]");
                }
                else {
                    buffer.append("null");
                }
            }
            else switch(value.getOfType()) {
                case Object:
                    ((RestThing)item).visit(this);
                    break;
                case List: {
                    if(item instanceof Thing) {
                        RestThing thing = (RestThing) item;
                        thing.visit(this);
                    }
                    else {
                        primitiveToJson(value.getOfType(),item);
                    }
                    break;
                }
                default: {
                    primitiveToJson(value.getOfType(),item);
                }
            }
            buffer.append(',').append('\n');
        }
        return false;
    }
    public void endVisitOfRestList(RestList value) {
        try {
            int index = buffer.lastIndexOf(",");
            if(index!=(-1)) {
                buffer.deleteCharAt(index); // trailing ','
            }
            tabs--;
            insertTabs();
            buffer.append(']');
            if (outStream != null) {
                outStream.write(buffer.toString().getBytes("UTF-8"));
                buffer = new StringBuilder();
            }
        }
        catch(IOException ex) {
            excep = new RestException("IO Exception", ex);
        }
    }
    /** Dictionary */
    public boolean beginVisitOfRestDictionary(RestDictionary value) {
        tabs++;
        buffer.append('{').append('\n');
        Iterator<String> iter = value.iteratorDict();
        insertTabs();
        while(iter.hasNext()) {
            String key = iter.next();
            buffer.append('"').append(key).append('"').append(':');
            Object item = value.getDict(key);
            switch(value.getOfType()) {
                case Object:
                    ((RestThing)item).visit(this);
                    break;
                case List: {
                    if(item instanceof Thing) {
                        RestThing thing = (RestThing) item;
                        thing.visit(this);
                    }
                    else {
                        primitiveToJson(value.getOfType(),item);
                    }
                    break;
                }
                default: {
                    primitiveToJson(value.getOfType(),item);
                }
            }
            buffer.append(',').append('\n');
            if(iter.hasNext())
                insertTabs();
        }
        return false;
    }
    public void endVisitOfRestDictionary(RestDictionary value) {
        try {
            int index = buffer.lastIndexOf(",");
            if(index!=(-1)) {
                buffer.deleteCharAt(index); // trailing ','
            }
            tabs--;
            insertTabs();
            buffer.append('}');
            if (outStream != null) {
                outStream.write(buffer.toString().getBytes("UTF-8"));
                buffer = new StringBuilder();
            }
        }
        catch(IOException ex) {
            excep = new RestException("IO Exception", ex);
        }
    }


    /** Member */
    public boolean beginVisitOfRestMember(RestMember value) {
        insertTabs();
        try {
            if (value.getParent().getType() != RestThingType.List) {
                buffer.append('"').append(value.getIdentifier()).append('"').append(": ");
            }
            if (value.getValue() == null) {
                if (value.getType() == RestThingType.Object) buffer.append("null");
                else if (value.getType() == RestThingType.List) buffer.append("[]");
                else buffer.append("null");
            } else if (value.getType() == RestThingType.Object) {
                RestObject obj = (RestObject) value.getValue();
                if (obj.isConcreteClass()) {
                    obj.visit(this);
                } else if (obj == null || obj.isEmpty()) {
                    buffer.append("{}");
                } else {
                    obj.visit(this);
                }
            } else if (value.getType() == RestThingType.Dictionary) {
                RestDictionary obj = (RestDictionary) value.getValue();
                if (obj == null || obj.isDictEmpty()) {
                    buffer.append("{}");
                } else {
                    obj.visit(this);
                }
            } else if (value.getType() == RestThingType.List && value.getValue() instanceof RestList) {
                RestList list = (RestList) value.getValue();
                if (list.isEmpty()) {
                    buffer.append("[]");
                } else {
                    list.visit(this);
                }
            } else if (value.getType() == RestThingType.List && !((List)value.getValue()).isEmpty() && (((List)value.getValue()).get(0) instanceof RestObject) ) {
                /** Special case where list is not RestList */
                List<RestObject> list = ((List<RestObject>)value.getValue());
                RestObject restObject = (RestObject)list.get(0);
                RestList restList = new RestList(value.getIdentifier(),restObject.getType(),restObject.getParent());
                for(RestObject ro : list) {
                    restList.add(ro);
                }
                restList.visit(this);
            } else if (value.getType() == RestThingType.List) {
                List list = (List) value.getValue();
                if (list.isEmpty()) {
                    buffer.append("[]");
                } else {
                    buffer.append('[');
                    for (Object obj : list) {
                        primitiveToJson(RestThingType.valueOf(obj), obj);
                    }
                    buffer.append(']');
                }
            } else {
                primitiveToJson(value.getType(), value.getValue());
            }
        }
        catch(Exception ex) {
            ex.printStackTrace();
        }
        buffer.append(',').append('\n');
        return false;
    }
    private boolean primitiveToJson(RestThingType type, Object item) {
        switch(type) {
            case Integer:
            case Long:
            case Short:
            case Boolean:
            case Decimal:
            case Double:
            case BigDecimal:
            case Byte:
                buffer.append(item.toString());
                break;
            case String:
            case Character:
                buffer.append('"').append(item.toString()).append('"');
                break;
            case Date: {
                Date date = (Date) item;
                buffer.append('"').append(ISO8601DateTimeSingleton.getInstance().getIsoDate(date)).append('"');
                break;
            }
            case Timestamp: {
                Timestamp ts = (Timestamp) item;
                buffer.append('"').append(ISO8601DateTimeSingleton.getInstance().getIsoDate(ts)).append('"');
                break;
            }
            case Enumeration:
                buffer.append('"').append(item.toString()).append('"');
                break;
            case Clob:
            case Blob:
                this.excep = new RestException("Type not implemented!" + RestThingType.Clob, new NotImplementedException());
                return false;
            default: {
                this.excep =  new RestException("Unknown Type=" + type.toString() + "  " + item);
                return false;
            }
        }
        return true;
    }
    public void insertTabs() {
        for(int ii=0; ii<tabs; ii++) buffer.append('\t');
    }
}
