package com.objectriver.microservices.util;
/**
 *                             NOTICE
 *               COPYRIGHT (c) 2016 ObjectRiver Inc
 *                UNPUBLISHED - ALL RIGHTS RESERVED
 */

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.Map;
import java.util.Random;

/**
 * Generate random data for testing purposes.
 */
public class RandomData {
    /** Random, initialied from RandomSingleton to insure that all
     * instances of this class generates a different sequence.
     */
    private Random rand = new Random( RandomSingleton.getInstance().nextLong());

    /** starting offset for random Timestamps */
    private long baseTime = System.currentTimeMillis();

    /** determine how often nulls are generated for nullable types 
     *  can also be set by the setNullRange method */
    private static final int NULLRANGE = 2;
    private int nullRange = NULLRANGE;

    /** table for computing ranges of integral and double types */
    private static final int[] exp10 = new int[] {
        1, 10, 100 };

    /** randomly select null or not null */
    private boolean getNull(boolean notnull) {
        return notnull ? false : nextInt(nullRange) == 1;
    }

    /** compute range of integral and double types */
    private long getLongRange(int places) {
        // keep within range
        if (places > 18)
            places = 18;

        if (places < exp10.length)
            return (long) exp10[places];
        long rval  = getLongRange(places / 2);

        rval *= rval;
        if ( (Math.abs(places) % 2) == 1 )
             rval *= 10;
        return rval;
    }

    private int getIntRange( int places ) {
        if (places > 9)
            places = 9;
        return (int) getLongRange(places);
    }

    private int getShortRange( int places ) {
        int range = getIntRange(places);
        if (range > Short.MAX_VALUE)
            range = Short.MAX_VALUE;
        return range;
    }

    private int getByteRange( int places ) {
        int range = getIntRange(places);
        if (range > Byte.MAX_VALUE)
            range = Byte.MAX_VALUE;
        return range;
    }

    private int nextInt(int range) {
        return rand.nextInt(range);
    }

    private long nextLong(long range) {
        long rval = rand.nextLong();
        if (rval < 0) rval *= -1;
        rval %= range ;
        return rval;
    }

    public Boolean nextBoolean(boolean notnull) {
        return getNull(notnull) ? null : Boolean.valueOf( rand.nextBoolean() );
    }

    // use lower-case in next* names to match with compiler basetype tokens.
    public Byte nextbyte(boolean notnull, int places) {
        return getNull(notnull) ? null : Byte.valueOf( (byte) nextInt( getByteRange(places)) );
    }

    public Long nextlong(boolean notnull, int places) {
        return getNull(notnull) ? null : Long.valueOf( nextLong( getLongRange(places)) );
    }

    public Integer nextint(boolean notnull, int places) {
        return getNull(notnull) ? null : Integer.valueOf( nextInt( (int) getIntRange(places) ) );
    }

    public Short nextshort(boolean notnull, int places) {
        return getNull(notnull) ? null : Short.valueOf( (short) nextInt( (int) getShortRange(places) ) );
    }

    public Character nextCharacter(boolean notnull) {
        return getNull(notnull) ? null : Character.valueOf((char) (65 + nextInt(26)));
    }

    /** Timestamp */
    public Timestamp nextTimestamp(boolean notnull) {
        if (getNull(notnull))
            return null;
        final int range = 1000000;
        long atime = baseTime + nextInt(range) - (range / 2);
        return new Timestamp(atime);
    }

    public String nextString(boolean notnull, int length) {
        final int MAX_STRING_LENGTH = 8000;

        if (getNull(notnull))
            return null;
        if (length > MAX_STRING_LENGTH)
            length = MAX_STRING_LENGTH;

        // random length, ave 75% of total length
        int useLen = 1 + ((1 + nextInt(3)) * ((length - 1) / 3));
        String rval = null;
        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < useLen; i++) {
            int r = 65 + nextInt(26);
            buffer.append( (char) r);
        }
        rval = buffer.toString();
        return rval;
    }

    public <E extends Enum<E>> E nextEnum( boolean notnull, E value, Enum[] values) {
        if (getNull(notnull))
            return null;
        int position = nextInt(10)%values.length;
        value = (E)values[position];
        return value;
    }
    public Object nextMapValue( boolean notnull, Map map) {
        if (getNull(notnull))
            return null;
        Object[] keys = map.keySet().toArray();
        Object key = keys[nextInt(keys.length)];
        return map.get(key);
    }

    public BigDecimal nextDecimal( boolean notnull, int precision, int scale) {
        if (getNull(notnull))
            return null;
        return BigDecimal.valueOf( nextLong( getLongRange(precision) ), scale );
    }

	public int getNullRange() {
		return nullRange;
	}

	public void setNullRange(int nullRange) {
		this.nullRange = nullRange;
	}
}
