package com.objectriver.runtime.webrpc.json;

import com.objectriver.runtime.webrpc.exception.WebRpcMarshallingException;
import com.objectriver.runtime.webrpc.json.marshals.JsonMarshal;

import javax.json.Json;
import javax.json.JsonValue;
import javax.json.stream.JsonParser;
import java.io.*;
import java.math.BigDecimal;

/**
 * ObjectRiver Inc, http://www.objectriver.net/
 * Copyright (c) 2002-2013, ObjectRiver(tm). All rights reserved.
 */
public class JSON {
    public JsonParserPlus jParser;
    public Writer writer =null;
    public Reader reader =null;

    public enum op {
        JSON_ENCODE,
        JSON_DECODE,
        JSON_FREE
    };
    private enum NUMERIC_TYPE {
        json_int,
        json_long,
        json_float,
        json_double
    }

    private JSON(){}
    /**
      * String
      */
    public JSON(String input) {
        jParser = new JsonParserPlus(Json.createParser(new StringReader(input)));
        x_op = op.JSON_DECODE;
    }
    public JSON(StringWriter output) {
        writer = output;
        x_op = op.JSON_ENCODE;
    }

    /**
     * TextStream
     */
    public JSON(Reader sr) {
        jParser = new JsonParserPlus(Json.createParser(sr));
        reader = sr;
        x_op = op.JSON_DECODE;

    }
    public JSON(Writer sw) {
        writer = sw;
        x_op = op.JSON_ENCODE;
    }

    public void close() throws IOException {
        writer.close();
    }

    public op x_op;
    public op getX_op() {
        return x_op;
    }
    public void setX_op(op x_op) {
        this.x_op = x_op;
    }

    public int version;
    public int getVersion() {
        return version;
    }
    public void setVersion(int version) {
        this.version = version;
    }

    private int nTabs = 0;
    private static final String tabs = "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t";
    public void startDocument()  throws WebRpcMarshallingException  {
        try {
            switch(x_op) {
            case JSON_ENCODE:
                writer.append("{\n");
                nTabs++;
                break;
            case JSON_DECODE:
                if(jParser.hasNext()) {
                    JsonParser.Event event = jParser.next();
                    if(event == JsonParser.Event.START_OBJECT)
                        return;
                }
                throw new WebRpcMarshallingException("JSON startDocument() not found?");
            }
        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("JSON START_OBJECT()",ioex);
        }
    }
    public void endDocument()  throws WebRpcMarshallingException {
        try {
            switch(x_op) {
            case JSON_ENCODE:
                writer.append("}\n");
                nTabs--;
                break;
            case JSON_DECODE:
                if(jParser.hasNext()) {
                    JsonParser.Event event = jParser.next();
                    if(event == JsonParser.Event.END_OBJECT)
                        return;
                }
                throw new WebRpcMarshallingException("JSON END_DOCUMENT() not found?");
            }
        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("endDocument()",ioex);
        }
    }
    public void startObject(String key)  throws WebRpcMarshallingException {
        try {
            switch(x_op) {
            case JSON_ENCODE:
                /**
                 * Handle Objects in arrays.
                 */
                if(key==null) {
                    writer.append('{');
                    /**
                     * lost one nTabs++
                     */
                    nTabs++;
                    return;
                }
                StringBuilder builder = new StringBuilder().append('"').append(key).append('"').append(": {\n");
                String str = builder.toString();
                writer.append(tabs.substring(0, nTabs));
                writer.append(str);
                nTabs++;
                break;
            case JSON_DECODE:
                if(jParser.hasNext()) {
                    try {
                        JsonParserPlus.Quad quad = jParser.pop();
                        if(quad.event == JsonParser.Event.START_OBJECT) {
                            /**
                             * Handle Objects in arrays.
                             */
                            if(key==null) {
                                return;
                            }
                            if(!quad.keyname.equals(key)) {
                                throw new WebRpcMarshallingException("JSON startObject() exspecting " + key + " got " + quad.keyname );
                            }
                            return;
                        }
                    }
                    catch(JsonParserException ex) {
                        throw new WebRpcMarshallingException("JSON startObject() Exception",ex);
                    }
                }
                throw new WebRpcMarshallingException("JSON startObject() not found?");
            }
        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("startObject()",ioex);
        }
    }
    public void endObject()  throws WebRpcMarshallingException {
        try {
            switch(x_op) {
            case JSON_ENCODE:
                writer.append(tabs.substring(0, nTabs));
                writer.append("}\n");
                nTabs--;
                break;
            case JSON_DECODE:
                if(jParser.hasNext()) {
                    JsonParser.Event event = jParser.next();
                    if(event == JsonParser.Event.END_OBJECT)
                        return;
                }
                throw new WebRpcMarshallingException("JSON END_OBJECT() not found?");
            }
        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("endObject()",ioex);
        }
    }
    public void startArray(String key)  throws WebRpcMarshallingException {
        try {
            switch(x_op) {
            case JSON_ENCODE:
                StringBuilder builder = new StringBuilder().append('"').append(key).append('"').append(": [");
                String str = builder.toString();
                writer.append(tabs.substring(0, nTabs));
                writer.append(str);
                nTabs++;
                break;
            case JSON_DECODE:
                if(jParser.hasNext()) {
                    try {
                        JsonParserPlus.Quad quad = jParser.pop();
                        if(quad.event == JsonParser.Event.START_ARRAY) {
                            if(!quad.keyname.equals(key)) {
                                throw new WebRpcMarshallingException("JSON startArray() exspecting " + key + " got " + quad.keyname );
                            }
                            return;
                        }
                    }
                    catch(JsonParserException ex) {
                        throw new WebRpcMarshallingException("JSON startArray() Exception",ex);
                    }
                }
                throw new WebRpcMarshallingException("JSON startArray() not found?");
             }
        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("startArray()",ioex);
        }
    }
    public void endArray()  throws WebRpcMarshallingException {
        try {
            switch(x_op) {
            case JSON_ENCODE:
                writer.append(tabs.substring(0, nTabs));
                writer.append("]\n");
                nTabs--;
                break;
            case JSON_DECODE:
                if(jParser.hasNext()) {
                    JsonParser.Event event = jParser.next();
                    if(event == JsonParser.Event.END_ARRAY)
                        return;
                }
                throw new WebRpcMarshallingException("JSON END_ARRAY() not found?");
             }
        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("endArray()",ioex);
        }
    }
    public void separator() throws WebRpcMarshallingException  {
        try {
            switch(x_op) {
            case JSON_ENCODE:
                writer.append(',');
                break;
             }
        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("separator()",ioex);
        }
    }
    public void put_nullArrayElement() throws WebRpcMarshallingException {
        try {
            writer.append(JsonValue.NULL.toString());
        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("put_nullArrayElement()",ioex);
        }
    }
    public void put_nullElement(String key) throws WebRpcMarshallingException {
        try {
            StringBuilder buffer = new StringBuilder();
            buffer.append('"').append(key).append('"').append(":");
            buffer.append(JsonValue.NULL.toString());
            String str = buffer.toString();
            writer.append(tabs.substring(0, nTabs));
            writer.append(str);
        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("put_nullElement()",ioex);
        }
    }
    public void put_nullObject(String key) throws WebRpcMarshallingException {
        try {
            StringBuilder buffer = new StringBuilder();
            buffer.append('"').append(key).append('"').append(":{}");
            String str = buffer.toString();
            writer.append(tabs.substring(0, nTabs));
            writer.append(str);

        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("put_nullObject()",ioex);
        }
    }
    public void put_nullObjectInArray() throws WebRpcMarshallingException {
        try {
            String str ="{}";
            writer.append(tabs.substring(0, nTabs));
            writer.append(str);

        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("put_nullObject()",ioex);
        }
    }
    public void put_nullArray(String key) throws WebRpcMarshallingException {
        try {
            StringBuilder buffer = new StringBuilder();
            buffer.append('"').append(key).append('"').append(":[]");
            String str = buffer.toString();
            writer.append(tabs.substring(0, nTabs));
            writer.append(str);

        }
        catch(IOException ioex) {
            throw new WebRpcMarshallingException("put_nullObject()",ioex);
        }
    }

    public void put_int(String key, Integer value)  throws WebRpcMarshallingException {
            try {
                StringBuilder buffer = new StringBuilder();
                if(JsonMarshal.arrayelement!=key) { // igonre key if array
                    buffer.append('"').append(key).append('"').append(":");
                }
                if(value==null) {
                    buffer.append(JsonValue.NULL.toString());
                }
                else {
                    buffer.append(value);
                }
                String str = buffer.toString();
                writer.append(tabs.substring(0, nTabs));
                writer.append(str);

            }
            catch(IOException ioex) {
                throw new WebRpcMarshallingException("put_int()",ioex);
            }
    }
    public Integer get_int(String key) throws WebRpcMarshallingException   {
        return (Integer)get_number(key,NUMERIC_TYPE.json_int);
    }

    public void put_long(String name, Long value) throws WebRpcMarshallingException   {
        try {
             StringBuilder buffer = new StringBuilder();
             if(JsonMarshal.arrayelement!=name) { // igonre key if array
                buffer.append('"').append(name).append('"').append(":");
             }
             if(value==null) {
                 buffer.append(JsonValue.NULL.toString());
             }
             else {
                 buffer.append(value);
             }
             String str = buffer.toString();
             writer.append(tabs.substring(0, nTabs));
             writer.append(str);

         }
         catch(IOException ioex) {
             throw new WebRpcMarshallingException("put_long()",ioex);
         }
    }
    public Long get_long(String key) throws WebRpcMarshallingException   {
        return (Long)get_number(key,NUMERIC_TYPE.json_long);
    }

    public void put_float(String name, Float value) throws WebRpcMarshallingException   {
        try {
             StringBuilder buffer = new StringBuilder();
             if(JsonMarshal.arrayelement!=name) { // igonre key if array
                buffer.append('"').append(name).append('"').append(":");
             }
             if(value==null) {
                 buffer.append(JsonValue.NULL.toString());
             }
             else {
                 buffer.append(value);
             }
             String str = buffer.toString();
             writer.append(tabs.substring(0, nTabs));
             writer.append(str);

         }
         catch(IOException ioex) {
             throw new WebRpcMarshallingException("put_float()",ioex);
         }
    }
    public Float get_float(String key) throws WebRpcMarshallingException   {
        return (Float)get_number(key,NUMERIC_TYPE.json_float);
    }

    public void put_double(String name, Double value) throws WebRpcMarshallingException   {
        try {
             StringBuilder buffer = new StringBuilder();
             if(JsonMarshal.arrayelement!=name) { // igonre key if array
                buffer.append('"').append(name).append('"').append(":");
             }
             if(value==null) {
                 buffer.append(JsonValue.NULL.toString());
             }
             else {
                 buffer.append(value);
             }
             String str = buffer.toString();
             writer.append(tabs.substring(0, nTabs));
             writer.append(str);

         }
         catch(IOException ioex) {
             throw new WebRpcMarshallingException("put_double()",ioex);
         }
    }
    public Double get_double(String key) throws WebRpcMarshallingException   {
        return (Double)get_number(key,NUMERIC_TYPE.json_double);
    }

    private Object get_number(String key, NUMERIC_TYPE nt) throws WebRpcMarshallingException {
        try {
             if(jParser.hasNext()) {
                 JsonParserPlus.Quad jEvent = jParser.pop();
                 switch(jEvent.event) {
                 case KEY_NAME:
                     if(!jEvent.keyname.equals(key)) {
                         throw new WebRpcMarshallingException( nt.toString() + "(). key mismatch. expecting " + key + " and got " + jEvent.keyname );
                     }
                     switch(jEvent.subevent) {
                     case VALUE_NUMBER: {
                         return getQuadValue(nt, jEvent);
                     }
                     case VALUE_NULL:
                         return null;
                     default:
                         throw new WebRpcMarshallingException(nt.toString() + "() type mismatch.");
                     }
                 /**
                  * Array members.
                  */
                 case VALUE_NUMBER:
                     if(jEvent.keyname!=JsonMarshal.arrayelement) {
                         throw new WebRpcMarshallingException(nt.toString() + "(). Unexpected json event " + jEvent.event.toString());
                     }
                     return getQuadValue(nt, jEvent);
                 case VALUE_NULL:
                     if(jEvent.keyname!=JsonMarshal.arrayelement) {
                         throw new WebRpcMarshallingException(nt.toString() + "(). Unexpected json event " + jEvent.event.toString());
                     }
                     return null;
                 default:
                     throw new WebRpcMarshallingException(nt.toString() + "(). Expected KEY_NAME got " + jEvent.event.toString());
                 }
             }
             else {
                 throw new WebRpcMarshallingException(nt.toString() + "(). No more JSON events?");
             }
         }
         catch(JsonParserException ex) {
             System.out.println(ex.getMessage());
         }
         throw new WebRpcMarshallingException("nt.toString() + \"(). Should never be here");
    }
    private Object getQuadValue(NUMERIC_TYPE nt, JsonParserPlus.Quad jEvent) throws WebRpcMarshallingException {
        BigDecimal big = (BigDecimal) jEvent.value;
        switch(nt) {
        case json_int:
            return big.intValue();
        case json_long:
            return big.longValue();
        case json_float:
            return big.floatValue();
        case json_double:
            return big.doubleValue();
        default:
            throw new WebRpcMarshallingException(nt.toString() + "(). Should never be here");
        }
    }

    public void put_bool(String key, Boolean value)  throws WebRpcMarshallingException {
            try {
                StringBuilder buffer = new StringBuilder();
                if(JsonMarshal.arrayelement!=key) { // igonre key if array
                    buffer.append('"').append(key).append('"').append(":");
                }
                if(value==null) {
                    buffer.append(JsonValue.NULL.toString());
                }
                else {
                    if(value.booleanValue())  {
                        buffer.append(JsonValue.TRUE.toString());
                    }
                    else {
                        buffer.append(JsonValue.FALSE.toString());
                    }
                }
                String str = buffer.toString();
                writer.append(tabs.substring(0, nTabs));
                writer.append(str);

            }
            catch(IOException ioex) {
                throw new WebRpcMarshallingException("get_bool()",ioex);
            }
    }
    public Boolean get_bool(String key) throws WebRpcMarshallingException   {
        try {
             if(jParser.hasNext()) {
                 JsonParserPlus.Quad jEvent = jParser.pop();
                 switch(jEvent.event) {
                 case KEY_NAME:
                     if(!jEvent.keyname.equals(key)) {
                         throw new WebRpcMarshallingException("put_string(). key mismatch. expecting " + key + " and got " + jEvent.keyname );
                     }
                     switch(jEvent.subevent) {
                     case VALUE_TRUE: {
                         return Boolean.TRUE;
                     }
                     case VALUE_FALSE: {
                          return Boolean.FALSE;
                      }
                     case VALUE_NULL:
                         return null;
                     default:
                         throw new WebRpcMarshallingException("get_bool() type mismatch.");
                     }
                 /**
                  * Array members.
                  */
                 case VALUE_TRUE: {
                     if(jEvent.keyname!=JsonMarshal.arrayelement) {
                         throw new WebRpcMarshallingException("get_bool(). Unexpected json event " + jEvent.event.toString());
                     }
                     return Boolean.TRUE;
                 }
                 case VALUE_FALSE: {
                     if(jEvent.keyname!=JsonMarshal.arrayelement) {
                         throw new WebRpcMarshallingException("get_bool(). Unexpected json event " + jEvent.event.toString());
                     }
                     return Boolean.FALSE;
                 }
                 case VALUE_NULL:
                     if(jEvent.keyname!=JsonMarshal.arrayelement) {
                         throw new WebRpcMarshallingException("get_bool(). Unexpected json event " + jEvent.event.toString());
                     }
                     return null;
                 default:
                     throw new WebRpcMarshallingException("get_bool(). Expected KEY_NAME got " + jEvent.event.toString());
                 }
             }
             else {
                 throw new WebRpcMarshallingException("get_bool(). No more JSON events?");
             }
         }
         catch(JsonParserException ex) {
             System.out.println(ex.getMessage());
         }
         throw new WebRpcMarshallingException("get_bool(). Should never be here");
    }

    public void put_string(String name, String value) throws WebRpcMarshallingException   {
        try {
             StringBuilder buffer = new StringBuilder();
             if(JsonMarshal.arrayelement!=name) { // igonre key if array
                buffer.append('"').append(name).append('"').append(":");
             }
             if(value==null) {
                 buffer.append(JsonValue.NULL.toString());
             }
             else {
                 buffer.append('"').append(value).append('"');
             }
             String str = buffer.toString();
             writer.append(tabs.substring(0, nTabs));
             writer.append(str);

         }
         catch(IOException ioex) {
             throw new WebRpcMarshallingException("put_string()",ioex);
         }
    }
    public String get_string(String key) throws WebRpcMarshallingException   {
        try {
             if(jParser.hasNext()) {
                 JsonParserPlus.Quad jEvent = jParser.pop();
                 switch(jEvent.event) {
                 case KEY_NAME:
                     if(jEvent.keyname.equals("null") && key == null){

                     }else{
                         if(!jEvent.keyname.equals(key)) {
                             throw new WebRpcMarshallingException("put_string(). key mismatch. expecting " + key + " and got " + jEvent.keyname );
                         }
                     }
                     switch(jEvent.subevent) {
                     case VALUE_STRING: {
                         return (String)jEvent.value;
                     }
                     case VALUE_NULL:
                         return null;
                     default:
                         throw new WebRpcMarshallingException("put_string() type mismatch.");
                     }
                 /**
                  * Array members.
                  */
                 case VALUE_STRING:
                 case VALUE_NULL:
                     if(jEvent.keyname!=JsonMarshal.arrayelement) {
                         throw new WebRpcMarshallingException("put_string(). Unexpected json event " + jEvent.event.toString());
                     }
                     return (String)jEvent.value;
                 default:
                     throw new WebRpcMarshallingException("put_string(). Expected KEY_NAME got " + jEvent.event.toString());
                 }
             }
             else {
                 throw new WebRpcMarshallingException("put_string(). No more JSON events?");
             }
         }
         catch(JsonParserException ex) {
             System.out.println(ex.getMessage());
         }
         throw new WebRpcMarshallingException("get_string(). Should never be here");
    }
    public void put_bytes(byte[] value, int length) throws WebRpcMarshallingException   {
    }
    public byte[] get_bytes(byte[] value, int length) throws WebRpcMarshallingException   {
        return value;
    }
    public void put_char(String name, Character value) throws WebRpcMarshallingException   {
        try {
             StringBuilder buffer = new StringBuilder();
             if(JsonMarshal.arrayelement!=name) { // igonre key if array
                buffer.append('"').append(name).append('"').append(":");
             }
             if(value==null) {
                 buffer.append(JsonValue.NULL.toString());
             }
             else {
                 buffer.append('"').append(value).append('"');
             }
             String str = buffer.toString();
             writer.append(tabs.substring(0, nTabs));
             writer.append(str);

         }
         catch(IOException ioex) {
             throw new WebRpcMarshallingException("put_char()",ioex);
         }
    }
    public Character get_char(String key) throws WebRpcMarshallingException   {
        String str = get_string(key);
        return Character.valueOf(str.charAt(0));
    }
}
