package com.objectriver.runtime.webrpc.json.marshals;

import com.objectriver.runtime.util.ArrayHelper;
import com.objectriver.runtime.util.ArrayIterator;
import com.objectriver.runtime.util.asserts.Assert;
import com.objectriver.runtime.webrpc.exception.WebRpcMarshallingException;
import com.objectriver.runtime.webrpc.json.*;

import javax.json.stream.JsonParser;
import java.lang.reflect.Array;
import java.util.LinkedList;


/**
 * ObjectRiver Inc, http://www.objectriver.net/
 * Copyright (c) 2002-2013, ObjectRiver(tm). All rights reserved.
 */
public abstract class JsonMarshal implements JsonProcT {
    protected JsonMarshal() {}

    private static final byte[] zeros = new byte[] {(byte) 0x00,(byte) 0x00,(byte) 0x00,(byte) 0x00};
    /*
     * json_pointer():
     *
     * JSON a pointer to a possibly recursive data structure. This
     * differs with json_reference in that it can serialize/deserialiaze
     * trees correctly.
     *
     *  What's sent is actually a union:
     *
     *  union object_pointer switch (boolean b) {
     *  case TRUE: object_data data;
     *  case FALSE: void nothing;
     *  }
     *
     * > objpp: Pointer to the pointer to the object.
     * > obj_size: size of the object.
     * > json_obj: routine to JSON an object.
     *
     */
    public boolean json_pointer(JSON jsons,JsonThing thing,JsonProcT proc) throws WebRpcMarshallingException {
        if (jsons.x_op == JSON.op.JSON_ENCODE) {
            if(thing.isNull()) {
                if(!proc.isBean()) {
                    jsons.put_nullElement(thing.getKey());
                }
            }
            else {
                return proc.execute(jsons,thing);
            }
        }
        if (jsons.x_op == JSON.op.JSON_DECODE) {
            try {
                if(jsons.jParser.hasNext()) {
                    JsonParserPlus.Quad jEvent = jsons.jParser.pop();
                    /**
                     * Adjust for Array
                     */
                    if(jEvent.keyname==arrayelement) {
                        jEvent.subevent=jEvent.event;
                        jEvent.event= JsonParser.Event.KEY_NAME;
                    }
                    switch(jEvent.event) {
                    case KEY_NAME:
                        /**
                         * When we have a bean, and we get a keyname, we just need to push and call the proc_t
                         */
                        if(!proc.isBean() && !jEvent.keyname.equals(thing.getKey())) {
                            throw new WebRpcMarshallingException("json_pointer(). key mismatch. expecting " + thing.getKey() + " and got " + jEvent.keyname );
                        }
                        /**
                         * Check for null array object {}
                         */
                        if(jEvent.subevent== JsonParser.Event.START_OBJECT && jsons.jParser.peek().event==JsonParser.Event.END_OBJECT) {
                            jsons.jParser.pop();
                            thing.setThing(null);
                        }
                        /**
                         * Handle NULL primitive
                         */
                        else if(jEvent.value==null && jEvent.subevent==null) {
                            thing.setThing(null);
                        }
                        else {
                            jsons.jParser.push(jEvent);
                            return proc.execute(jsons,thing);
                        }
                        break;
                    case START_OBJECT:
                        jsons.jParser.push(jEvent);
                        return proc.execute(jsons,thing);
                    case START_ARRAY:
                        jsons.jParser.push(jEvent);
                        return proc.execute(jsons,thing);
                    case END_OBJECT:
                        thing.setThing(null);
                        jsons.jParser.push(jEvent);
                        break;
                    default:
                        throw new WebRpcMarshallingException("json_pointer(). Expected KEY_NAME,START_OBJECT,START_ARRAY got " + jEvent.event.toString());
                    }
                }
                else {
                    throw new WebRpcMarshallingException("json_pointer(). No more JSON events?");
                }
            }
            catch(JsonParserException ex) {
                throw new WebRpcMarshallingException("json_pointer(). " + ex.getMessage(), ex);
            }
        }
        return true;
    }
    /*
     * JSON an indirect pointer
     * json_reference is for recursively translating a structure that is
     * referenced by a pointer inside the structure that is currently being
     * translated.  pp references a pointer to storage. If *pp is null
     * the  necessary storage is allocated.
     * size is the sizeof the referneced structure.
     * proc is the routine to handle the referenced structure.
     */
    public boolean json_reference(JSON jsons, JsonThing thing, JsonProcT proc)  throws WebRpcMarshallingException {
        return proc.execute(jsons,thing);
    }
    /*
     * JSON opaque data
     * Allows the specification of a fixed size sequence of opaque bytes.
     * cp points to the opaque object and cnt gives the byte length.
     */
    public boolean json_opaque(JSON jsons, byte[] bytes, int cnt)  throws WebRpcMarshallingException {
        if(cnt==0) {
            return true;
        }

        /**
         * Marshall
         */
        try {
            if (jsons.x_op == JSON.op.JSON_DECODE) {
                jsons.get_bytes(bytes, cnt);
            }
            else if (jsons.x_op == JSON.op.JSON_ENCODE) {
                jsons.put_bytes(bytes, cnt);
            }
        }
        finally{}
        return true;
    }

    /*
     * json_vector():
     *
     * JSON a fixed length array. Unlike variable-length arrays,
     * the storage of fixed length arrays is static and unfreeable.
     * > basep: base of the array
     * > size: size of the array
     * > elemsize: size of each element
     * > json_elem: routine to JSON each element
     */
    public boolean
    json_vector(JSON jsons, JsonThing thing1, int nelem, JsonProcT proc) throws WebRpcMarshallingException  {
        int i;
        if(jsons.x_op== JSON.op.JSON_ENCODE) {
            Object vector[] = (Object[])thing1.getThing();
            /*
             * check for null, if xdr_nullable() is not present in encode.
             */
            if(!(proc instanceof json_nullable) ) {
                for (i = 0; i < nelem ; i++) {
                    if(vector[i]==null) {
                        throw new WebRpcMarshallingException("json_vector() class " + proc.getClazz().getName() + " type can not be null! Check webrpc definition.");
                    }
                }
            }
            JsonThing thing2 = new JsonThing();
            try {
                jsons.startArray(thing1.getKey());
                for (i = 0; i < nelem; i++) {
                    Object elptr=vector[i];
                    if(elptr==null) {
                        jsons.put_nullArrayElement();
                    }
                    else if(!proc.execute(jsons,thing2.setJvalue(arrayelement,elptr)))  {
                        return false;
                    }
                    if(i+1<nelem) {
                        jsons.separator();
                    }
                }
                jsons.endArray();
            }
            finally{}
        }
        else if(jsons.x_op== JSON.op.JSON_DECODE) {
            Object vector[] = (Object[]) Array.newInstance(proc.getClazz(), nelem);
            JsonThing thing2 = new JsonThing();
            try {
                jsons.startArray(thing1.getKey());
                for (i = 0; i < nelem; i++) {
                    JsonParserPlus.Quad arrayMember = jsons.jParser.popArrayElement();
                    /**
                     * Push at arraymember back into stack with constant keyname
                     * json array members do not have names.
                     */
                    arrayMember.keyname=JsonMarshal.arrayelement;
                    jsons.jParser.push(arrayMember);
                    /**
                     * retrieve element and add to list.
                     */
                    if(!proc.execute(jsons,thing2.setJvalue(JsonMarshal.arrayelement,null))) {
                        return false;
                    }
                    vector[i]=thing2.getThing();
                    if(i+1<nelem) {
                        jsons.separator();
                    }
                }
                thing1.setThing(vector);
                jsons.endArray();
            }
            catch(JsonParserException ex) {
                throw new WebRpcMarshallingException("json_vector(). " + ex.getMessage(), ex);
            }
        }
        return(true);
    }

    /*
     * JSON an array of arbitrary elements
     * *addrp is a pointer to the array, *sizep is the number of elements.
     * If addrp is NULL (*sizep * elsize) bytes are allocated.
     * elsize is the size (in bytes) of each element, and elproc is the
     * json procedure to call to handle each element of the array.
     *
     *  register JSON *jsons;
     *  caddr_t *addrp;		// array pointer
     *  u_int *sizep;		// number of elements
     *  u_int maxsize;		// max numberof elements
     *  u_int elsize;		// size in bytes of each element
     *  jsonproc_t elproc;	// json routine to handle each element
     */
    public boolean
    json_array(JSON jsons, JsonThing thing1, JsonProcT proc) throws WebRpcMarshallingException  {
        return json_array(jsons, thing1, Integer.MAX_VALUE, proc);
    }

    public boolean
    json_array(JSON jsons, JsonThing thing, int maxsize, JsonProcT proc) throws WebRpcMarshallingException  {
        if (jsons.x_op == JSON.op.JSON_ENCODE) {
            if(thing.isNull()) {
                jsons.put_nullArray(thing.getKey());
            }
            else {
                Object[] array = (Object[])thing.getThing();
                /*
                 * check for null, if xdr_nullable() is not present in encode.
                 */
                if(!(proc instanceof json_nullable) ) {
                    for (int i = 0; i < array.length ; i++) {
                        if(array[i]==null) {
                            throw new WebRpcMarshallingException("json_array() class " + proc.getClazz().getName() + " type can not be null! Check webrpc definition.");
                        }
                    }
                }

                jsons.startArray(thing.getKey());
                ArrayIterator iter = new ArrayIterator(array);
                while(iter.hasNext()) {
                    Object object = iter.next();
                    if(object==null) {
                        if(proc.isBean()){
                            jsons.put_nullObjectInArray();
                        }else{
                            jsons.put_nullArrayElement();
                        }
                    }
                    else {
                        if(proc.isBean()) jsons.startObject(null);
                        if(!proc.execute(jsons,new JsonThing(arrayelement,object))) {
                            return false;
                        }
                        if(proc.isBean()) jsons.endObject();
                    }
                    if(iter.hasNext()) {
                        jsons.separator();
                    }
                }
                jsons.endArray();
                return true;
            }
        }
        else if (jsons.x_op == JSON.op.JSON_DECODE) {
            try {
                LinkedList<Object> list = new LinkedList<Object>();
                if(jsons.jParser.hasNext()) {
                    JsonParserPlus.Quad jEvent = jsons.jParser.pop();
                    switch(jEvent.event) {
                    case START_ARRAY:
                        if(!jEvent.keyname.equals(thing.getKey())) {
                            throw new WebRpcMarshallingException("json_array(). key mismatch. expecting " + thing.getKey() + " and got " + jEvent.keyname );
                        }
                        /**
                         * Check for null array
                         */
                        JsonParserPlus.Quad subEvent = jsons.jParser.popArrayElement();
                        if(subEvent.value== JsonParser.Event.END_ARRAY) {
                            // empty array.
                            Object[] array = ArrayHelper.recast(proc.getClazz(),new Object[0]);
                            thing.setThing(array);
                        }
                        else {
                            jsons.jParser.push(subEvent);
                            WhileLoop:
                            while(true) {
                                /**
                                 * Peek at arraymember to see if its the end of the array.
                                 */
                                JsonParserPlus.Quad arrayMember = jsons.jParser.popArrayElement();
                                if(arrayMember.event==JsonParser.Event.END_ARRAY) {
                                    break WhileLoop;
                                }
                                /**
                                 * Push at arraymember back into stack with constant keyname
                                 * json array members do not have names.
                                 */
                                arrayMember.keyname=JsonMarshal.arrayelement;
                                jsons.jParser.push(arrayMember);
                                /**
                                 * retrieve element and add to list.
                                 */
                                if(proc.isBean()) jsons.startObject(null);

                                proc.execute(jsons,thing.setJvalue(JsonMarshal.arrayelement,null));

                                if(proc.isBean()) jsons.endObject();
                                list.add(thing.getThing());
                            }
                            thing.setThing(ArrayHelper.recast(proc.getClazz(),list.toArray()));
                        }
                        break;
                    case KEY_NAME:
                        Assert.fail("implement START_OBJECT json_array");
                        break;
                    case START_OBJECT:
                        Assert.fail("implement START_ARRAY json_array");
                        break;
                    default:
                        throw new WebRpcMarshallingException("json_array(). Expected KEY_NAME,START_OBJECT,START_ARRAY got " + jEvent.event.toString());
                    }
                }
                else {
                     throw new WebRpcMarshallingException("json_array(). No more JSON events?");
                }
            }
            catch(JsonParserException ex) {
                throw new WebRpcMarshallingException("json_array(). " + ex.getMessage(), ex);
            }
        }
        return true;
    }
    /*
     * JSON an array of bytes.
     * This routine front-ends json_opaque, by passing the size like json_string.
     * */
    public boolean json_bytes(JSON jsons, JsonThing thing, JsonProcT proc) throws WebRpcMarshallingException {
        Integer size=0;
        byte[] bytes=null;
        /**
         * First deal with length of string.
         */
        try {
            if (jsons.x_op == JSON.op.JSON_ENCODE) {
                bytes = (byte[])thing.getThing();
                size = ( bytes==null )? 0 : bytes.length;
            }
            // ENCODE DECODE
            JsonThing thing2 = new JsonThing("size",size);
            if(!json_int.instance.execute(jsons,thing2)) {
                return false;
            }
            size = (Integer)thing2.getThing();
            /**
             * Now deal with marshalling string.
             */
            switch(jsons.x_op) {
            case JSON_DECODE:
                if(size==0) {
                    thing.setThing(null);
                    return true;
                }
                byte[] decoded = new byte[size];
                if(json_opaque(jsons, decoded ,size )) {
                    thing.setThing(decoded);
                    return true;
                }
                return false;
            case JSON_ENCODE: {
                if(size==0) {
                    return true;
                }
                return json_opaque(jsons, bytes, size);
            }
            }
        }
        finally {}
        return false;
    }

    public abstract Class getClazz();

    public boolean isBean() {
        return false;
    }

    public static final String objectelement = "oBjEcTElEmEnT";
    public static final String arrayelement = "jSoNaRrAyElEmEnT";
}

