package com.objectriver.runtime.webrpc.xdr.marshals;

import com.objectriver.runtime.util.ArrayHelper;
import com.objectriver.runtime.webrpc.exception.WebRpcMarshallingException;
import com.objectriver.runtime.webrpc.xdr.XDR;
import com.objectriver.runtime.webrpc.xdr.XdrProcT;
import com.objectriver.runtime.webrpc.xdr.XdrThing;

import java.io.IOException;
import java.lang.reflect.Array;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;

/**
 * ObjectRiver Inc, http://www.objectriver.net/
 * Copyright (c) 2002-2013, ObjectRiver(tm). All rights reserved.
 */
public abstract class XdrMarshal implements XdrProcT {
    private static final byte[] zeros = new byte[] {(byte) 0x00,(byte) 0x00,(byte) 0x00,(byte) 0x00};
    /*
     * xdr_pointer():
     *
     * XDR a pointer to a possibly recursive data structure. This
     * differs with xdr_reference in that it can serialize/deserialiaze
     * trees correctly.
     *
     *  What's sent is actually a union:
     *
     *  union object_pointer switch (boolean b) {
     *  case TRUE: object_data data;
     *  case FALSE: void nothing;
     *  }
     *
     * > objpp: Pointer to the pointer to the object.
     * > obj_size: size of the object.
     * > xdr_obj: routine to XDR an object.
     *
     */
    public boolean xdr_pointer(XDR xdrs,XdrThing thing,XdrProcT proc) throws WebRpcMarshallingException {
        Boolean more_data = (thing.getThing()!=null);
        XdrThing moreDataThing = new XdrThing(more_data); // encode default
        try {
            if(!xdr_bool.instance.execute(xdrs,moreDataThing)) { // place on wire or decode
                return false;
            }
            more_data = (Boolean)moreDataThing.getThing(); // replace for decode.
            if(!more_data) {
                thing.setThing(null);
                return true;
            }
            return xdr_reference(xdrs,thing,~0,proc); // encode/decode reference
        }
        finally{}
    }
    /*
     * XDR an indirect pointer
     * xdr_reference is for recursively translating a structure that is
     * referenced by a pointer inside the structure that is currently being
     * translated.  pp references a pointer to storage. If *pp is null
     * the  necessary storage is allocated.
     * size is the sizeof the referneced structure.
     * proc is the routine to handle the referenced structure.
     */
    public boolean xdr_reference(XDR xdrs, XdrThing thing, int size, XdrProcT proc)  throws WebRpcMarshallingException {
        return proc.execute(xdrs,thing);
    }
    /*
     * XDR opaque data
     * Allows the specification of a fixed size sequence of opaque bytes.
     * cp points to the opaque object and cnt gives the byte length.
     */
    public boolean xdr_opaque(XDR xdrs, byte[] bytes, int cnt)  throws WebRpcMarshallingException {
        if(cnt==0) {
            return true;
        }
        /**
         * Round byte count to full xdr units.
         */
        int rndup = cnt % XDR.BYTES_PER_XDR_UNIT;
        if(rndup>0) rndup = XDR.BYTES_PER_XDR_UNIT - rndup;

        /**
         * Marshall
         */
        try {
            if (xdrs.x_op == XDR.op.XDR_DECODE) {
                xdrs.get_bytes(bytes, cnt);
                if (rndup == 0)
                    return (true);
                xdrs.get_bytes(new byte[XDR.BYTES_PER_XDR_UNIT], rndup);
            }
            else if (xdrs.x_op == XDR.op.XDR_ENCODE) {
                xdrs.put_bytes(bytes, cnt);
                if (rndup == 0)
                    return (true);
                xdrs.put_bytes(zeros, rndup);
            }
        }
        finally {}
        return true;
    }

    /*
     * xdr_vector():
     *
     * XDR a fixed length array. Unlike variable-length arrays,
     * the storage of fixed length arrays is static and unfreeable.
     * > basep: base of the array
     * > size: size of the array
     * > xdr_elem: routine to XDR each element
     */
    public boolean
    xdr_vector(XDR xdrs, XdrThing thing1, int nelem, XdrProcT proc) throws WebRpcMarshallingException  {
        int i;
        Object vector[] = (Object[])thing1.getThing();
        if(xdrs.x_op== XDR.op.XDR_DECODE) {
            vector = (Object[])Array.newInstance(proc.getClazz(), nelem);
        }
        /*
         * check for null, if xdr_nullable() is not present in encode.
         */
        else if(xdrs.x_op== XDR.op.XDR_ENCODE && !(proc instanceof xdr_nullable) ) {
            for (i = 0; i < nelem ; i++) {
                if(vector[i]==null) {
                    throw new WebRpcMarshallingException("xdr_vector() class " + proc.getClazz().getName() + " type can not be null! Check webrpc definition.");
                }
            }
        }

        XdrThing thing2 = new XdrThing();
        try {
            for (i = 0; i < nelem; i++) {
                Object elptr=vector[i];
                if (! proc.execute(xdrs, thing2.setThing(elptr))) {
                    return(false);
                }
                if(xdrs.x_op== XDR.op.XDR_DECODE) {
                    vector[i]=thing2.getThing();
                }
            }
            if(xdrs.x_op== XDR.op.XDR_DECODE) {
                thing1.setThing(vector);
            }
        }
        finally{}
        return(true);
    }

    /*
     * XDR an array of arbitrary elements
     * *addrp is a pointer to the array, *sizep is the number of elements.
     * If addrp is NULL (*sizep * elsize) bytes are allocated.
     * elsize is the size (in bytes) of each element, and elproc is the
     * xdr procedure to call to handle each element of the array.
     *
     *  register XDR *xdrs;
     *  caddr_t *addrp;		// array pointer
     *  u_int *sizep;		// number of elements
     *  u_int maxsize;		// max numberof elements
     *  u_int elsize;		// size in bytes of each element
     *  xdrproc_t elproc;	// xdr routine to handle each element
     */
    public boolean
    xdr_array(XDR xdrs, XdrThing thing1, int maxsize, XdrProcT proc) throws WebRpcMarshallingException  {
        /**
         * handle byte[] arrays special
         */
        if(proc==xdr_byte.instance) {
            return xdr_bytes(xdrs,thing1);
        }
        /**
         * xdr_array
         */
        XdrThing thing2 = new XdrThing();
        int i;
        Object[] array = (Object[])thing1.getThing();
        int sizep = (array==null) ? 0 : array.length;
        int count;  /* the actual element count */
        boolean stat = true;

        try {
            /* like strings, arrays are really counted arrays */
            if (!xdr_int.instance.execute(xdrs, thing2.setThing(sizep))) {
                return (false);
            }
            sizep=(Integer)thing2.getThing();
            count = sizep;
            if ((count > maxsize) && (xdrs.x_op != XDR.op.XDR_FREE)) {
                return (false);
            }

            /*
             * if we are deserializing, we may need to allocate an array.
             * We also save time by checking for a null array if we are freeing.
             */
            if (thing1.isNull())
                switch (xdrs.x_op) {
                case XDR_DECODE:
                    if (count == 0) {
                        array = ArrayHelper.recast(proc.getClazz(),new Object[0]);
                        thing1.setThing(array);
                        return (true);
                    }
                    array = new Object[count];
                    break;
            }
            /*
             * check for null, if xdr_nullable() is not present in encode.
             */
            if(xdrs.x_op== XDR.op.XDR_ENCODE && !(proc instanceof xdr_nullable) ) {
                for (i = 0; i < count ; i++) {
                    if(array[i]==null) {
                        throw new WebRpcMarshallingException("xdr_array() class " + proc.getClazz().getName() + " type can not be null! Check webrpc definition.");
                    }
                }
            }

            /*
             * now we xdr each element of array
             */
            for (i = 0; (i < count) && stat; i++) {
                stat = proc.execute(xdrs, thing2.setThing(array[i]));
                if(!true) return false; // bail out on failure.
                if (xdrs.x_op == XDR.op.XDR_DECODE) {
                    array[i]=thing2.getThing();
                }
                /*
                 *      (wdk): next line avoids implicit pointer arithmetic.
                 *		target = (caddr_t)((int)target + elsize);
                 */
            }
            if (xdrs.x_op == XDR.op.XDR_DECODE) {
                array = ArrayHelper.recast(proc.getClazz(),array);
                thing1.setThing(array);
            }
        }
        finally{}
        return (stat);
    }
    /*
     * XDR an array of bytes.
     * This routine front-ends xdr_opaque, by passing the size like xdr_string.
     * */
    public boolean xdr_bytes(XDR xdrs, XdrThing thing) throws WebRpcMarshallingException {
        Integer size=0;
        byte[] bytes=null;
        /**
         * First deal with length of string.
         */
        try {
            if (xdrs.x_op == XDR.op.XDR_ENCODE) {
                bytes = (byte[])thing.getThing();
                size = ( bytes==null )? 0 : bytes.length;
            }
            // ENCODE DECODE
            XdrThing thing2 = new XdrThing(size);
            if(!xdr_int.instance.execute(xdrs,thing2)) {
                return false;
            }
            size = (Integer)thing2.getThing();
            /**
             * Now deal with marshalling string.
             */
            switch(xdrs.x_op) {
            case XDR_DECODE:
                if(size==0) {
                    thing.setThing(null);
                    return true;
                }
                byte[] decoded = new byte[size];
                if(xdr_opaque(xdrs, decoded ,size )) {
                    thing.setThing(decoded);
                    return true;
                }
                return false;
            case XDR_ENCODE: {
                if(size==0) {
                    return true;
                }
                return xdr_opaque(xdrs, bytes, size);
            }
            }
        }
        catch (BufferUnderflowException ex) {
            throw new WebRpcMarshallingException("xdr_bytes", ex);
        }
        catch (BufferOverflowException ex) {
            throw new WebRpcMarshallingException("xdr_bytes", ex);
        }
        return false;
    }
    public abstract Class getClazz();
}
